#include <cinttypes>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <pthread.h>
#include <cassert>
#include <chrono>
#include <iostream>

#include "globals.h"
#include "utils/graphIO.h"
#include "utils/debug.h"
#include "utils/diffAnalysis.h"

#include "representations/tradListGraphR.h"
#include "representations/metisR.h"
#include "representations/diffGraphR.h"

#include "coders/myVarintByteBasedCoder.h"

#include "offsetStructures/offsetsBV_RRR.h"

using namespace std;

std::string GRAPH_S;
std::string GRAPH_FILE_T;
std::string GRAPH_FILE_R;
std::string dir;
std::string file;
ofstream f_cat;

int MAX_ITERATIONS = 10000;
int ITERATIONS_PER_WARMUP = 100;
int MICRO_SECONDS = 1000000;

TradListGraphR* rep_original_t = NULL;
MetisR* rep_original_m = NULL;
SimpleRecGraphR* rep_bisected_r = NULL;

DiffGraphR<OffsetsBV_RRR, MyVarintByteBasedCoder>* dif_B;


bool hack = true;

std::chrono::time_point<std::chrono::system_clock> s, e, start_time, end_time;


void measure_deg_dif_B(int part_level, v_id* rand_v) {
  f_cat.open(dir + "/" + file + "_dif_deg_" + to_string(part_level), ios::app);
  for (int i = 0; i < ITERATIONS_PER_WARMUP; i++) {
    v_id deg = dif_B->getVertexDegree(rand_v[i]);
  }
  for (int i = ITERATIONS_PER_WARMUP; i < ITERATIONS_PER_WARMUP + MAX_ITERATIONS; i++) {
    start_time = std::chrono::high_resolution_clock::now();
    v_id deg = dif_B->getVertexDegree(rand_v[i]);
    end_time = std::chrono::high_resolution_clock::now();
    std::chrono::duration<double> elapsed_seconds = end_time-start_time;
    f_cat << GRAPH_S << " deg [us] " << elapsed_seconds.count()*MICRO_SECONDS << endl;
  } 
  f_cat.close();
}

void measure_N_dif_B(int part_level, v_id* rand_v) {
  f_cat.open(dir + "/" + file + "_dif_N_" + to_string(part_level), ios::app);
  for (int i = 0; i < ITERATIONS_PER_WARMUP; i++) {
    v_id* N = dif_B->getVertexNeighbors(rand_v[i]);
  }
  for (int i = ITERATIONS_PER_WARMUP; i < ITERATIONS_PER_WARMUP + MAX_ITERATIONS; i++) {
    start_time = std::chrono::high_resolution_clock::now();
    v_id* N = dif_B->getVertexNeighbors(rand_v[i]);
    end_time = std::chrono::high_resolution_clock::now();
    std::chrono::duration<double> elapsed_seconds = end_time-start_time;
    f_cat << GRAPH_S << " neigh [us] " << elapsed_seconds.count()*MICRO_SECONDS << endl;
  } 
  f_cat.close();
}



int main (int argc, char *argv[]) {
  GRAPH_S = argv[1];
  GRAPH_FILE_T = argv[2];
  GRAPH_FILE_R = argv[3];
  dir = argv[4];
  file = argv[5];
  assert(argc == 6);

  srand(time(NULL));

  if(GRAPH_FILE_T == "" || GRAPH_FILE_R == "") {
    return EXIT_FAILURE;
  }

  cout << "Loading original adjacency list from file...   ";
  GraphIO::loadGraph(GRAPH_FILE_T.c_str(), &rep_original_t, &rep_original_m);
  cout << "Done" << endl;

  cout << ">>> Loading bisected representation from file." << endl;
  rep_bisected_r = GraphIO::readRecursivePartitioningFromFile(GRAPH_FILE_R);
  cout << "Done" << endl;

  for (int i=32; i>1; i-=2) {

    f_cat.open(dir + "/" + file + "_size", ios::app);

    cout << "Constructing dif_B for level " << i << endl;
    s = std::chrono::high_resolution_clock::now();
    dif_B = new DiffGraphR<OffsetsBV_RRR, MyVarintByteBasedCoder>(DEGREE_HIGH_TO_LOW, rep_bisected_r, rep_original_t, i);
    e = std::chrono::high_resolution_clock::now();    
    chrono::duration<double> cnt = e - s; 
    f_cat << GRAPH_S << " construction of RB for level " << i << ": " << cnt.count() << "s" << endl;
    cout << "Done" << endl;

    double dif_B_ADJ_SIZE = dif_B->adj_data_total_size_in_bytes()/(1.0*1024*1024);
    double dif_B_OFFSETS_SIZE = dif_B->offsets_total_size_in_bytes()/(1.0*1024*1024);
    double dif_B_TOTAL_SIZE = dif_B_ADJ_SIZE + dif_B_OFFSETS_SIZE; 
    double dif_B_TOTAL_REDUN = dif_B->adj_data_total_redundancy_in_bytes()/(1.0*1024*1024);
    double dif_B_SIZE_NO_REDUN = dif_B_TOTAL_SIZE - dif_B_TOTAL_REDUN;

    f_cat << "RB adj [MB] " << dif_B_ADJ_SIZE << endl;
    f_cat << "RB off [MB] " << dif_B_OFFSETS_SIZE << endl;
    f_cat << "RB tot [MB] " << dif_B_TOTAL_SIZE << endl;
    f_cat << "RB red [MB] " << dif_B_TOTAL_REDUN << endl;
    f_cat << "RB nor [MB] " << dif_B_SIZE_NO_REDUN << endl;
    
    f_cat.close();

    v_id* rand_v = new v_id[MAX_ITERATIONS + ITERATIONS_PER_WARMUP]();

    for(int64_t i = 0; i < (MAX_ITERATIONS + ITERATIONS_PER_WARMUP); ++i) {
      rand_v[i] = rand() % rep_original_t->n_;
    }

    cout << "Measuring deg for dif_B...   ";
    measure_deg_dif_B(i, rand_v);
    cout << "Done" << endl;
    cout << "Measuring neigh for dif_B...   ";
    measure_N_dif_B(i, rand_v);
    cout << "Done" << endl;

    delete [] rand_v;
    delete dif_B;

  }
  
  delete rep_original_t;
  delete rep_original_m;

  return EXIT_SUCCESS;
}

