#if !defined(__DIFF_GRAPH_R_H__)
#define      __DIFF_GRAPH_R_H__

#include "globals.h"
#include "utils/debug.h"
#include "representations/misc.h"
#include "representations/graphR.h"
#include "representations/metisR.h"
#include "representations/tradListGraphR.h"
#include "permuters/recPermuter.h"


/*********** TEMPLATE CLASS *************/

template<typename OffsetsType, typename CoderType>
class DiffGraphR : public GraphR {
  public:
    DiffGraphR(VertexOrder v_order, const SimpleRecGraphR* r, const TradListGraphR* t);
    DiffGraphR(VertexOrder v_order, SimpleRecGraphR* r, const TradListGraphR* t, bool hack);
    DiffGraphR(VertexOrder v_order, const SimpleRecGraphR* r, const MetisR* m); 
    DiffGraphR(VertexOrder v_order, v_id n, v_id m);
    ~DiffGraphR();

    inline v_id getVertexDegree(v_id v) __attribute__((always_inline));
    inline v_id* getVertexNeighborsAndDegree(v_id v, v_id* nr_neighs_out) __attribute__((always_inline));
    inline string* getVertexNeighborsStrAndDegree(v_id v, v_id* nr_neighs_out) __attribute__((always_inline));
    inline bool adjUseV1Neighbors(v_id v1, v_id v2) __attribute__((always_inline));
    inline bool adjUseV2Neighbors(v_id v1, v_id v2) __attribute__((always_inline));
    inline bool adjUseMinNeighbors(v_id v1, v_id v2) __attribute__((always_inline));

    inline uint64_t adj_data_total_size_in_bytes();
    inline uint64_t offsets_total_size_in_bytes();
    inline uint64_t helper_str_total_size_in_bytes();

    inline uint64_t adj_data_total_redundancy_in_bytes();
    inline uint64_t adj_data_edges_redundancy_in_bytes();
    inline uint64_t adj_data_signs_redundancy_in_bytes();
    inline uint64_t adj_data_neighbors_redundancy_in_bytes();

  private:
  public:
    void initializeParameters(VertexOrder v_order, const TradListGraphR* t);
    void initializeParameters(VertexOrder v_order, const MetisR* t);
    void initializeStructures(VertexOrder v_order, const SimpleRecGraphR* r, const TradListGraphR* t);
    void initializeStructures(VertexOrder v_order, SimpleRecGraphR* r, const TradListGraphR* t, bool hack);
    void initializeStructures(VertexOrder v_order, const SimpleRecGraphR* r, const MetisR* t);

    VertexOrder v_order_;

    // The main containers for graph data
    // TODO: non-pointer for performance?
    CoderType* coder_;
    RecPermuter* permuter_;
    OffsetsType* offsets_;  
};

/* Initialization functions */

template<typename OffsetsType, typename CoderType>
DiffGraphR<OffsetsType, CoderType>::DiffGraphR(VertexOrder v_order, const SimpleRecGraphR* r, const TradListGraphR* t) {
  initializeParameters(v_order, t);
  initializeStructures(v_order, r, t);
}

template<typename OffsetsType, typename CoderType>
DiffGraphR<OffsetsType, CoderType>::DiffGraphR(VertexOrder v_order, SimpleRecGraphR* r, const TradListGraphR* t, bool hack) {
  initializeParameters(v_order, t);
  initializeStructures(v_order, r, t, hack);
}

template<typename OffsetsType, typename CoderType>
DiffGraphR<OffsetsType, CoderType>::DiffGraphR(VertexOrder v_order, const SimpleRecGraphR* r, const MetisR* m) {
  initializeParameters(v_order, m);
  initializeStructures();
}

template<typename OffsetsType, typename CoderType>
void DiffGraphR<OffsetsType, CoderType>::initializeParameters(VertexOrder v_order, const TradListGraphR* t) {
  v_order_ = v_order;
  n_ = t->n_;
  m_ = t->m_;
}

template<typename OffsetsType, typename CoderType>
void DiffGraphR<OffsetsType, CoderType>::initializeParameters(VertexOrder v_order, const MetisR* m) {
  v_order_ = v_order;
  n_ = m->n_;
  m_ = m->m_;
}

template<typename OffsetsType, typename CoderType>
void DiffGraphR<OffsetsType, CoderType>::initializeStructures(VertexOrder v_order, const SimpleRecGraphR* r, const TradListGraphR* t) {
  permuter_ = new RecPermuter();
  TradListGraphR* permuted_adj_list = permuter_->permute(r, t, v_order);
  int64_t* offsets;
  coder_ = new CoderType();
  coder_->encode(permuted_adj_list, &offsets);
  offsets_ = new OffsetsType(permuted_adj_list, offsets);
  delete permuted_adj_list;
  delete [] offsets;
}

template<typename OffsetsType, typename CoderType>
void DiffGraphR<OffsetsType, CoderType>::initializeStructures(VertexOrder v_order, SimpleRecGraphR* r, const TradListGraphR* t, bool hack) {
  __UNUSED(hack);
  permuter_ = new RecPermuter();
  TradListGraphR* permuted_adj_list = permuter_->permute(r, t, v_order);
  int64_t* offsets;
  coder_ = new CoderType();
  coder_->encode(permuted_adj_list, r, &offsets);
  offsets_ = new OffsetsType(permuted_adj_list, offsets);
  delete permuted_adj_list;
  delete [] offsets;
}


template<typename OffsetsType, typename CoderType>
void DiffGraphR<OffsetsType, CoderType>::initializeStructures(VertexOrder v_order, const SimpleRecGraphR* r, const MetisR* m) {
  permuter_ = new RecPermuter();
  TradListGraphR* permuted_adj_list = permuter_->permute(r, m, v_order);
  int64_t* offsets;
  coder_ = new CoderType();
  coder_->encode(permuted_adj_list, &offsets);
  offsets_ = new OffsetsType(m, offsets);
  delete permuted_adj_list;
  delete [] offsets;
}

template<typename OffsetsType, typename CoderType>
DiffGraphR<OffsetsType, CoderType>::~DiffGraphR() {
  delete offsets_;
  delete coder_;
  delete permuter_;
}

/* Graph API functions */

template<typename OffsetsType, typename CoderType>
v_id DiffGraphR<OffsetsType, CoderType>::getVertexDegree(v_id v) {
  v_id offset = offsets_->getVertexOffset(v);
  v_id degree = coder_->decodeVertexDegree(v, offset);
  assert((uint64_t)degree < (uint64_t)n_);
  return degree;
}

template<typename OffsetsType, typename CoderType>
bool DiffGraphR<OffsetsType, CoderType>::adjUseV1Neighbors(v_id v1, v_id v2) {
  v_id v1_offset = offsets_->getVertexOffset(v1);
  return coder_->decodeVertexConnectionUseV1Neighbors(v1, v2, v1_offset);
}

template<typename OffsetsType, typename CoderType>
bool DiffGraphR<OffsetsType, CoderType>::adjUseV2Neighbors(v_id v1, v_id v2) {
  v_id v2_offset = offsets_->getVertexOffset(v2);
  return coder_->decodeVertexConnectionUseV2Neighbors(v1, v2, v2_offset);
}

template<typename OffsetsType, typename CoderType>
bool DiffGraphR<OffsetsType, CoderType>::adjUseMinNeighbors(v_id v1, v_id v2) {
  v_id v1_offset = offsets_->getVertexOffset(v1);
  v_id v2_offset = offsets_->getVertexOffset(v2);
  return coder_->decodeVertexConnectionUseMinNeighbors(v1, v2, v1_offset, v2_offset);
}

// NULL means zero neighbors
template<typename OffsetsType, typename CoderType>
v_id* DiffGraphR<OffsetsType, CoderType>::getVertexNeighborsAndDegree(v_id v, v_id* nr_neighs_out) {
  v_id offset = offsets_->getVertexOffset(v);
  v_id degree = coder_->decodeVertexDegree(v, offset);
  v_id* neighbors = coder_->decodeVertexNeighbors(v, offset);
  assert(degree <= n_);
  *nr_neighs_out = degree;
  return neighbors;
}

template<typename OffsetsType, typename CoderType>
string* DiffGraphR<OffsetsType, CoderType>::getVertexNeighborsStrAndDegree(v_id v, v_id* nr_neighs_out) {
  v_id offset = offsets_->getVertexOffset(v);
  v_id degree = coder_->decodeVertexDegree(v, offset);
  string* neighbors = coder_->decodeVertexNeighborsStr(v, offset);
  assert(degree <= n_);
  *nr_neighs_out = degree;
  return neighbors;
}


template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::adj_data_total_size_in_bytes() {
  return coder_->adj_data_total_size_in_bytes();
}

template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::offsets_total_size_in_bytes() {
  return offsets_->total_size_in_bytes();
}

template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::helper_str_total_size_in_bytes() {
  return sizeof(permuter_) + sizeof(*permuter_) + sizeof(coder_) + sizeof(*coder_) + sizeof(v_order_);
}

template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::adj_data_total_redundancy_in_bytes() {
  return coder_->adj_data_total_redundancy_in_bytes();
}

template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::adj_data_edges_redundancy_in_bytes() {
  return coder_->adj_data_edges_redundancy_in_bytes();
}

template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::adj_data_signs_redundancy_in_bytes() {
  return coder_->adj_data_signs_redundancy_in_bytes();
}

template<typename OffsetsType, typename CoderType>
uint64_t DiffGraphR<OffsetsType, CoderType>::adj_data_neighbors_redundancy_in_bytes() {
  return coder_->adj_data_neighbors_redundancy_in_bytes();
}

#endif
