#include <fstream>
#include <cassert>
#include <iostream>

#include "BV_Graph.h"

BV_Graph::BV_Graph(const char* file_name) {
	basename = string(file_name);
	readProperties();
	loadOffsets();
	adj = new InputBitStream((basename + GRAPH_EXTENSION).c_str());
	deg = new InputBitStream((basename + GRAPH_EXTENSION).c_str());
}

BV_Graph::~BV_Graph() {
	delete [] offsets;
	delete adj;
	delete deg;
}

BV_Graph::BV_Graph(string b, int n, long m, int ws, int mil, long *off, InputBitStream *a, InputBitStream *d) {
	basename = b;
	num_nodes = n;
	num_edges = m;
	window_size = ws;
	min_interval_length = mil;
	offsets = off;
	adj = a;
	deg = d;
}

void BV_Graph::resetPointers() {
	offsets = NULL;
	adj = NULL;
	deg = NULL;
}

BV_Graph* BV_Graph::copy() {
	return new BV_Graph(basename, num_nodes, num_edges, window_size, min_interval_length, offsets, adj->copy(), deg->copy());
}

void BV_Graph::readProperties() {
	ifstream prop_file;
	prop_file.open((basename + PROPERTIES_EXTENSION).c_str());
	string line;
	while (prop_file >> line) {
		if (line.compare(0,10,"windowsize") == 0) {
			string ws = line.substr(11,line.length()-11);
			window_size = stoi(ws);
		}
		else if (line.compare(0,5,"nodes") == 0) {
			string n = line.substr(6,line.length()-6);
			num_nodes = stoi(n);
		}
		else if (line.compare(0, 4, "arcs") == 0) {
			string m = line.substr(5,line.length()-5);
			num_edges = stoi(m);
		}
		else if (line.compare(0, 17, "minintervallength") == 0) {
			string mil = line.substr(18, line.length()-18);
			min_interval_length = stoi(mil);
		}

	}

}
	
void BV_Graph::loadOffsets() {
	offsets = new long[num_nodes+1];
	InputBitStream *off_ibs = new InputBitStream((basename + OFFSETS_EXTENSION).c_str());
	offsets[0] = readOffset(off_ibs);
	for (int i=1; i<num_nodes+1; ++i) {
		offsets[i] = readOffset(off_ibs) + offsets[i-1];
//		cout << "offset[" << i << "]: " << offsets[i] << endl;

	}
}

string BV_Graph::getBasename() {
	return basename;
}

int BV_Graph::getNumNodes() {
	return num_nodes;
}
	
long BV_Graph::getNumEdges() {
	return num_edges;
}

void BV_Graph::setPosition(InputBitStream *ibs, long pos) {
	ibs->SetPosition(pos);
}
	
long BV_Graph::readOffset(InputBitStream *off_ibs) {
	return off_ibs->ReadLongGamma(); 
}

int BV_Graph::readDegree(InputBitStream *adj_ibs) {
	return adj_ibs->ReadGamma();
}
	
int BV_Graph::readReference(InputBitStream *adj_ibs) {
	return adj_ibs->ReadUnary();
}
	
int BV_Graph::readBlockCount(InputBitStream *adj_ibs) {
	return adj_ibs->ReadGamma();
}
	
int BV_Graph::readBlock(InputBitStream *adj_ibs) {
	return adj_ibs->ReadGamma();
}

int BV_Graph::readResidual(InputBitStream *adj_ibs) {
	return adj_ibs->ReadZeta(DEFAULT_ZETA_K);
}

long BV_Graph::readLongResidual(InputBitStream *adj_ibs) {
	return adj_ibs->ReadLongZeta(DEFAULT_ZETA_K);
}

int BV_Graph::readIntervalCount(InputBitStream *adj_ibs) {
	return adj_ibs->ReadGamma();
}

int BV_Graph::readLeft(InputBitStream *adj_ibs) {
	return adj_ibs->ReadGamma();
}

long BV_Graph::readLongLeft(InputBitStream *adj_ibs) {
	return adj_ibs->ReadLongGamma();
}
	
int BV_Graph::readLen(InputBitStream *adj_ibs) {
	return adj_ibs->ReadGamma();
}


int BV_Graph::getVertexDegree(int v) {
	assert (v > -1 && v < num_nodes);
	setPosition(deg, offsets[v]);
	return readDegree(deg);
}

void BV_Graph::getVertexNeighbors(int v, vector<int> *neighs) {
	assert (v > -1 && v < num_nodes);
	int ref=0, block_count=0, extra_count=0, interval_count=0, residual_count=0;
	vector<int> block, left, len, residuals;
	setPosition(adj, offsets[v]);

	int degree = readDegree(adj);
	if (degree == 0) {
		return;
	}
	if (window_size > 0) {
		ref = readReference(adj);
	}
	else {
		ref = -1;
	}

	if (ref > 0) {
		block_count = readBlockCount(adj);
	
		int copied = 0, total = 0;
		for(int i = 0; i < block_count; i++ ) {
			block.push_back(readBlock(adj) + ( i == 0 ? 0 : 1 ));
			total += block[i];
			if (i % 2 == 0) copied += block[i];
		}
		if (block_count % 2 == 0)  {
			copied += getVertexDegree(v - ref) - total;
		}
		extra_count = degree - copied;
	}
	else {
		extra_count = degree;
	}

	if ( extra_count > 0 ) {

		if (min_interval_length != 0 && (interval_count = readIntervalCount(adj)) != 0 ) {
	
			int prev = 0;
			int tmp = readLongLeft(adj);
	
			left.push_back((int)((tmp % 2 == 0 ? tmp >> 1 : -( tmp >> 1 ) - 1) + v));
			
			prev = left[0];
			len.push_back(readLen(adj) + min_interval_length);
	
			prev += len[0];
			extra_count -= len[0];

			for (int i = 1; i < interval_count; i++ ) {
				left.push_back(readLeft(adj) + prev + 1);
				prev = left[i];
				len.push_back(readLen(adj) + min_interval_length);
				prev += len[i];
				extra_count -= len[i];

			}

		}
	}

	residual_count = extra_count;
	if (residual_count > 0) {
		int tmp = readLongResidual(adj);
		residuals.push_back((int)((tmp % 2 == 0 ? tmp >> 1 : -( tmp >> 1 ) - 1) + v));
		for (int i=1; i<residual_count; ++i) {
			residuals.push_back(residuals[i-1] + readResidual(adj) + 1);
		}
	}

	if (ref > 0) {
 		vector<int> *ref_neighs = new vector<int>();
		getVertexNeighbors(v-ref, ref_neighs);
		int k = 0;
		for (int i=0; i<block_count; ++i) {
			int j = 0;
			while (j < block[i]) {
				if (i % 2 == 0) {
					neighs->push_back((*ref_neighs)[k]);
				}
				++k;
				++j;
			}
		}
		if (block_count % 2 == 0) {
			for (int i=k; i<ref_neighs->size(); ++i) {
				neighs->push_back((*ref_neighs)[i]);
			}
		}
		delete ref_neighs;

	}
	if (interval_count > 0) {
		for (int i=0; i<interval_count; ++i) {
			for (int j=0; j<len[i]; ++j) {
				neighs->push_back(left[i]+j);
			}
		}
	}

	if (residual_count > 0) {
		for (int i=0; i<residual_count; ++i) {
			neighs->push_back(residuals[i]);
		}
	}


}