#include <cinttypes>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <pthread.h>
#include <cassert>
#include <chrono>
#include <iostream>
#include <fstream>

#include "BV_Graph.h"

using namespace std;

ofstream f_cat;

int MAX_ITERATIONS = 10000;
int ITERATIONS_PER_WARMUP = 100;
int MICRO_SECONDS = 1000000;
int MAX_THREADS = 18;

pthread_barrier_t thread_barrier;

typedef struct thread_params_t {
  int thread_id;
  int num_threads;
} thread_params;

BV_Graph *graph;

std::chrono::time_point<std::chrono::system_clock> start_time, end_time;


void measure_deg(int thread_id, int num_threads, int* rand_v, BV_Graph *g) {
  pthread_barrier_wait(&thread_barrier);
  for (int i = 0; i < ITERATIONS_PER_WARMUP; i++) {
    int deg = g->getVertexDegree(rand_v[i]);
  }
  pthread_barrier_wait(&thread_barrier);
  for (int i = ITERATIONS_PER_WARMUP; i < ITERATIONS_PER_WARMUP + MAX_ITERATIONS; i++) {
    if (thread_id == 0) {
      f_cat.open(g->getBasename() + "_deg_" + to_string(num_threads), ios::app);
      start_time = std::chrono::high_resolution_clock::now();
    }
    int deg = g->getVertexDegree(rand_v[i]);
    if (thread_id == 0) {
      end_time = std::chrono::high_resolution_clock::now();
      std::chrono::duration<double> elapsed_seconds = end_time-start_time;
      f_cat << elapsed_seconds.count()*MICRO_SECONDS << endl;
      f_cat.close();
    }
  } 
  
}

void measure_N(int thread_id, int num_threads, int* rand_v, BV_Graph *g) {
  pthread_barrier_wait(&thread_barrier);
  vector<int> *neighs = new vector<int>();
  for (int i = 0; i < ITERATIONS_PER_WARMUP; i++) {
    g->getVertexNeighbors(rand_v[i], neighs);
    neighs->clear();

  }
  pthread_barrier_wait(&thread_barrier);
  for (int i = ITERATIONS_PER_WARMUP; i < ITERATIONS_PER_WARMUP + MAX_ITERATIONS; i++) {
    if (thread_id == 0) {
      f_cat.open(g->getBasename() + "_neigh_"+ to_string(num_threads), ios::app);
      start_time = std::chrono::high_resolution_clock::now();
    }
    g->getVertexNeighbors(rand_v[i], neighs);
    if (thread_id == 0) {
      end_time = std::chrono::high_resolution_clock::now();
      std::chrono::duration<double> elapsed_seconds = end_time-start_time;
      f_cat << elapsed_seconds.count()*MICRO_SECONDS << endl;
      f_cat.close();
    }
    neighs->clear();
  } 
  
}


void* measure(void* args) {
  thread_params* params = (thread_params*)args;

  int thread_id = params->thread_id;
  int num_threads = params->num_threads;
  BV_Graph *g = graph->copy();
  int* rand_v = new int[MAX_ITERATIONS + ITERATIONS_PER_WARMUP]();


  for(int64_t i = 0; i < (MAX_ITERATIONS + ITERATIONS_PER_WARMUP); ++i) {
    rand_v[i] = rand() % g->getNumNodes();
  }

  if (thread_id == 0) {
    cout << "Measuring deg with " << num_threads << " threads" << endl;
  }
  measure_deg(thread_id, num_threads, rand_v, g);
  pthread_barrier_wait(&thread_barrier);

  if (thread_id == 0) {
    cout << "Measuring neigh with " << num_threads << " threads" << endl;
  }
  measure_N(thread_id, num_threads, rand_v, g);
  pthread_barrier_wait(&thread_barrier);

  free(params);

  g->resetPointers();
  delete g;
  delete [] rand_v;

  pthread_exit(NULL);
}


int main (int argc, char *argv[]) {
  
  graph = new BV_Graph(argv[1]);

  srand(time(NULL));

  for(int i = 1; i <= MAX_THREADS; i += 2) {
    pthread_attr_t attr;
    pthread_t* threads = new pthread_t[i];
    //for(int j = 0; j < i; ++j) {
    //  threads[j] = new pthread_t();
    //}
    void* status;

    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);  

    pthread_barrier_init(&thread_barrier, NULL, i);

    for(int j = 0; j < i; j++) {
      thread_params* params = (thread_params*) malloc(sizeof(thread_params));
      params->thread_id = j;
      params->num_threads = i;
      pthread_create(&threads[j], &attr, measure, (void*)params);
    }

    pthread_attr_destroy(&attr);

    for(int j = 0; j < i; j++) {
      pthread_join(threads[j], &status);
    }

    pthread_barrier_destroy(&thread_barrier);

    //for(int j = 0; j < i; ++j) {
    //  delete threads[j];
    //}
    delete [] threads;

    if (i == 1) {
      i = 0;
    }
  }

  delete graph;
  return EXIT_SUCCESS;
}

